const db = require('../config/db');
const ExcelJS = require('exceljs');

const exportRegistrations = async (req, res, next) => {
  const { type, eventId, search } = req.query;

  let query = `
    SELECT 
      r.id, 
      r.event_id, 
      e.title as event_title, 
      r.user_id, 
      r.registered_at, 
      r.status, 
      r.details,
      COALESCE(u.username, r.details->>'fullName', r.details->>'contactName', 'Guest') as username,
      COALESCE(u.email, r.details->>'email') as email
    FROM registrations r
    JOIN events e ON r.event_id = e.id
    LEFT JOIN users u ON r.user_id = u.id
    WHERE 1=1
  `;
  const params = [];
  let paramCount = 1;

  if (type) {
    query += ` AND r.details->>'formType' = $${paramCount}`;
    params.push(type);
    paramCount++;
  }

  if (eventId) {
    query += ` AND r.event_id = $${paramCount}`;
    params.push(eventId);
    paramCount++;
  }

  if (search) {
    query += ` AND (
      r.details->>'email' ILIKE $${paramCount} OR
      r.details->>'fullName' ILIKE $${paramCount} OR
      r.details->>'contactName' ILIKE $${paramCount} OR
      r.details->>'orgName' ILIKE $${paramCount} OR
      u.username ILIKE $${paramCount} OR
      u.email ILIKE $${paramCount}
    )`;
    params.push(`%${search}%`);
    paramCount++;
  }

  query += ` ORDER BY r.registered_at DESC`;

  try {
    const result = await db.query(query, params);
    const registrations = result.rows;

    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Registrations');

    // Define columns based on type or generic
    const columns = [
      { header: 'ID', key: 'id', width: 10 },
      { header: 'Event', key: 'event_title', width: 30 },
      { header: 'Name', key: 'username', width: 25 },
      { header: 'Email', key: 'email', width: 30 },
      { header: 'Status', key: 'status', width: 15 },
      { header: 'Registered At', key: 'registered_at', width: 20 },
    ];

    // Add dynamic columns based on details if needed, or just dump details
    // For better UX, let's extract common fields from details based on type
    if (type === 'participant') {
      columns.push({ header: 'Phone', key: 'phone', width: 15 });
      columns.push({ header: 'Organization', key: 'organization', width: 25 });
    } else if (type === 'intervenant') {
      columns.push({ header: 'Expertise', key: 'expertise', width: 25 });
      columns.push({ header: 'LinkedIn', key: 'linkedin', width: 30 });
    } else if (type === 'soutien') {
      columns.push({ header: 'Contact Name', key: 'contactName', width: 25 });
      columns.push({ header: 'Partnership Type', key: 'partnershipType', width: 20 });
    }

    worksheet.columns = columns;

    registrations.forEach(reg => {
      const row = {
        id: reg.id,
        event_title: reg.event_title,
        username: reg.username,
        email: reg.email,
        status: reg.status,
        registered_at: new Date(reg.registered_at).toLocaleString(),
        // Extract details
        phone: reg.details.phone,
        organization: reg.details.organization || reg.details.orgName,
        expertise: reg.details.expertise,
        linkedin: reg.details.linkedin,
        contactName: reg.details.contactName,
        partnershipType: reg.details.partnershipType
      };
      worksheet.addRow(row);
    });

    // Style header
    worksheet.getRow(1).font = { bold: true };
    worksheet.getRow(1).fill = {
      type: 'pattern',
      pattern: 'solid',
      fgColor: { argb: 'FFCCCCCC' }
    };

    res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    res.setHeader('Content-Disposition', `attachment; filename=registrations-${type || 'all'}-${Date.now()}.xlsx`);

    await workbook.xlsx.write(res);
    res.end();

  } catch (error) {
    next(error);
  }
};

module.exports = { exportRegistrations };
